using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Data;

namespace gov.va.med.vbecs.GUI.controls
{
	/// <summary>
	/// Class DataGridComboBox
	/// </summary>
	public class DataGridComboBox:ComboBoxLookup
	{
		private const int WM_LBUTTONDOWN = 513;	// 0x0201
		private const int WM_LBUTTONUP = 514;	// 0x0202
		private const int WM_KEYDOWN = 256;		// 0x0100
		private const int WM_KEYUP = 257;		// 0x0101
		private const int WM_CHAR = 0x0102;		// 258

		private string _valueMember;
		private string _displayMember;
		private DataTable _dataSource;
		
		/// <summary>
		///  Basic ComboBox used for all ComboBoxes displayes in a datagrid
		/// </summary>
		public DataGridComboBox()
		{
			this.DropDownStyle = ComboBoxStyle.DropDown;
			this.Visible = false;
		}

		private void InitializeComponent()
		{
		
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="dataSource"></param>
		/// <param name="displayMember"></param>
		/// <param name="valueMember"></param>
		public DataGridComboBox(DataTable dataSource, string displayMember , string valueMember):base()
		{
			this.SetDataSource(dataSource, displayMember , valueMember);
		}

		/// <summary>
		/// isInEditOrNavigateMode
		/// </summary>
		public bool isInEditOrNavigateMode = true;

		/// <summary>
		/// SetDataSource
		/// </summary>
		/// <param name="dataSource"></param>
		/// <param name="displayMember"></param>
		/// <param name="valueMember"></param>
		public void SetDataSource(DataTable dataSource, string displayMember , string valueMember)
		{
			this._dataSource = dataSource;
			this._displayMember = displayMember;
			this._valueMember = valueMember;
			//
			this.Items.Clear();
			//
			foreach(DataRow dr in this._dataSource.Rows)
			{
				this.Items.Add(dr[this._displayMember].ToString());
			}
		}

		/// <summary>
		/// Get SelectedValueMember
		/// </summary>
		public object SelectedValueMember
		{
			get
			{
				if (this.SelectedIndex < 0)
				{
					return null;
				}
				else
				{
					return this._dataSource.Rows[this.SelectedIndex][this._valueMember];
				}
			}
		}

		/// <summary>
		/// ProcessKeyPreview
		/// </summary>
		/// <param name="msg"></param>
		/// <returns></returns>
		protected override bool ProcessKeyPreview(ref Message msg)
		{
			if (msg.Msg == WM_KEYDOWN)
			{
				switch(msg.WParam.ToInt32())       
				{         
					case (int)Keys.Up:   
						break;                  
					case (int)Keys.Down:   
						break;
					case (int)Keys.Left:   
						break;
					case (int)Keys.Right:   
						break;
					default:
						break;
				}

			}
			//
			if (msg.Msg == WM_KEYUP)
			{
			}
			//
			return base.ProcessKeyPreview(ref msg);
		}
	}

	/// <summary>
	/// Class DataGridComboBoxColumn
	/// </summary>
	public class DataGridComboBoxColumn:DataGridColumnStyle
	{
		//
		// Creates a combo box column on a data grid
		// all cells in the column have the same data source


		//
		// UI constants    
		//
		private int _xMargin = 2;
		private int _yMargin = 1;
		private DataGridComboBox _combo;
		private bool _isFirstRowOnly = false;
		
		//
		// Used to track editing state
		//

		private string _oldVal=new string(string.Empty.ToCharArray());
		private bool _inEdit= false;

		/// <summary>
		/// Constructor
		/// Create a new column - DisplayMember, ValueMember
		/// Passed by ordinal
		/// </summary>
		/// <param name="dataSource"></param>
		/// <param name="displayMember"></param>
		/// <param name="valueMember"></param>
		public DataGridComboBoxColumn(DataTable dataSource, int displayMember,int valueMember):this(dataSource,dataSource.Columns[displayMember].ToString(),dataSource.Columns[valueMember].ToString())
		{
		}

		/// <summary>
		/// Constructor
		/// Create a new column - DisplayMember, ValueMember passed by string
		/// </summary>
		/// <param name="dataSource"></param>
		/// <param name="displayMember"></param>
		/// <param name="valueMember"></param>
		public DataGridComboBoxColumn(DataTable dataSource,string displayMember,string valueMember)
		{
			this.NullText = string.Empty;
			//
			_combo = new DataGridComboBox(dataSource, displayMember, valueMember);
			_combo.Visible = false;
		}

		/// <summary>
		/// Constructor
		/// </summary>
		public DataGridComboBoxColumn()
		{
			this.NullText = string.Empty;
			//
			_combo = new DataGridComboBox();
			_combo.Visible = false;
		}

		/// <summary>
		/// Get ComboBox
		/// </summary>
		public DataGridComboBox ComboBox
		{
			get
			{
				return this._combo;
			}
		}

		/// <summary>
		/// Get/Set IsFirstRowOnly
		/// </summary>
		public bool IsFirstRowOnly
		{
			get
			{
				return this._isFirstRowOnly;
			}
			set
			{
				this._isFirstRowOnly = value;
			}
		}

		//------------------------------------------------------
		// Methods overridden from DataGridColumnStyle
		//------------------------------------------------------
		
		/// <summary>
		/// Abort Changes
		/// </summary>
		/// <param name="rowNum"></param>
		protected override void Abort(int rowNum)
		{
			System.Diagnostics.Debug.WriteLine("Abort()");
			RollBack();
			HideComboBox();
			EndEdit();
		}
		
		/// <summary>
		/// Commit Changes
		/// </summary>
		/// <param name="dataSource"></param>
		/// <param name="rowNum"></param>
		/// <returns></returns>
		protected override bool Commit(CurrencyManager dataSource,int rowNum)
		{
			HideComboBox();
			if(!_inEdit)
			{
				return true;
			}
			try
			{
				object val = _combo.Text;
				if(NullText.Equals(val))
				{
					val = System.Convert.DBNull; 
				}
				SetColumnValueAtRow(dataSource, rowNum, val);
			}
			catch
			{
				RollBack();
				return false;	
			}
			
			this.EndEdit();
			return true;
		}

		/// <summary>
		/// ConcedeFocus
		/// </summary>
		protected override void ConcedeFocus()
		{
			this._combo.Visible = false;
		}

		/// <summary>
		///  Edit Grid
		/// </summary>
		/// <param name="source"></param>
		/// <param name="rowNum"></param>
		/// <param name="bounds"></param>
		/// <param name="readOnly"></param>
		/// <param name="instantText"></param>
		/// <param name="cellIsVisible"></param>
		protected override void Edit(CurrencyManager source ,int rowNum, Rectangle bounds, bool readOnly, string instantText, bool cellIsVisible)
		{
			_combo.Text = string.Empty;
			Rectangle originalBounds = bounds;
			_oldVal = _combo.Text;

			if (this.IsFirstRowOnly && rowNum > 0)
			{
				return;
			}

			if(cellIsVisible)
			{
				bounds.Offset(_xMargin, _yMargin);
				bounds.Width -= _xMargin * 2;
				bounds.Height -= _yMargin;
				_combo.Bounds = bounds;
				_combo.Visible = true;
			}
			else
			{
				_combo.Bounds = originalBounds;
				_combo.Visible = false;
			}
			
			_combo.Text = GetText(GetColumnValueAtRow(source, rowNum));
			
			if(instantText!=null)
			{
				_combo.Text = instantText;
			}
			_combo.RightToLeft = this.DataGridTableStyle.DataGrid.RightToLeft;
			_combo.Focus();
			
			if(instantText==null)
			{
				_combo.SelectAll();
				
				// Pre-selects an item in the combo when user tries to add
				// a new record by moving the columns using tab.

				// Combo.SelectedIndex = 0;
			}
			else
			{
				int end = _combo.Text.Length;
				_combo.Select(end, 0);
			}
			if(_combo.Visible)
			{
				DataGridTableStyle.DataGrid.Invalidate(originalBounds);
			}

			_inEdit = true;
		}

		/// <summary>
		/// GetMinimumHeight
		/// </summary>
		/// <returns></returns>
		protected override int GetMinimumHeight()
		{
			//
			// Set the minimum height to the height of the combobox
			//
			return _combo.PreferredHeight + _yMargin;
		}

		/// <summary>
		/// GetPreferredHeight
		/// </summary>
		/// <param name="g"></param>
		/// <param name="val"></param>
		/// <returns></returns>
		protected override int GetPreferredHeight(Graphics g ,object val)
		{
			System.Diagnostics.Debug.WriteLine("GetPreferredHeight()");
			int newLineIndex  = 0;
			int newLines = 0;
			string valueString = this.GetText(val);
			do
			{
				newLineIndex = valueString.IndexOf("r\n", newLineIndex + 1);
				newLines += 1;
			}
			while(newLineIndex != -1);
			//
			return this.FontHeight * newLines + _yMargin;
		}

		/// <summary>
		/// Size
		/// </summary>
		/// <param name="g"></param>
		/// <param name="val"></param>
		/// <returns></returns>
		protected override Size GetPreferredSize(Graphics g, object val)
		{
			Size extents = Size.Ceiling(g.MeasureString(GetText(val), this.DataGridTableStyle.DataGrid.Font));
			extents.Width += _xMargin * 2 + DataGridTableGridLineWidth ;
			extents.Height += _yMargin;
			return extents;
		}

		/// <summary>
		/// Paint
		/// </summary>
		/// <param name="g"></param>
		/// <param name="bounds"></param>
		/// <param name="source"></param>
		/// <param name="rowNum"></param>
		protected override void Paint(Graphics g, Rectangle bounds, CurrencyManager source, int rowNum)
		{
			Paint(g, bounds, source, rowNum, false);
		}

		/// <summary>
		/// Paint
		/// </summary>
		/// <param name="g"></param>
		/// <param name="bounds"></param>
		/// <param name="source"></param>
		/// <param name="rowNum"></param>
		/// <param name="alignToRight"></param>
		protected override void Paint(Graphics g, Rectangle bounds, CurrencyManager source, int rowNum, bool alignToRight)
		{
			string text = GetText(GetColumnValueAtRow(source, rowNum));
			PaintText(g, bounds, text, alignToRight);
		}

		/// <summary>
		/// Paint
		/// </summary>
		/// <param name="g"></param>
		/// <param name="bounds"></param>
		/// <param name="source"></param>
		/// <param name="rowNum"></param>
		/// <param name="backBrush"></param>
		/// <param name="foreBrush"></param>
		/// <param name="alignToRight"></param>
		protected override void Paint(Graphics g, Rectangle bounds, CurrencyManager source, int rowNum, Brush backBrush, Brush foreBrush, bool alignToRight)
		{
			if ((this.ReadOnly)||(this.IsFirstRowOnly && rowNum > 0))
			{
				backBrush = Brushes.LightGray;
				foreBrush = Brushes.LightGray;
			}
			//
			string text = GetText(GetColumnValueAtRow(source, rowNum));
			PaintText(g, bounds, text, backBrush, foreBrush, alignToRight);
		}

		/// <summary>
		/// SetDataGridInColumn
		/// </summary>
		/// <param name="val"></param>
		protected override void SetDataGridInColumn(DataGrid val)
		{
			base.SetDataGridInColumn(val);
			if(_combo.Parent!=val)
			{
				if(_combo.Parent!=null)
				{
					_combo.Parent.Controls.Remove(_combo);
				}
			}
			if(val!=null) 
			{
				val.Controls.Add(_combo);
			}
		}

		/// <summary>
		/// UpdateUI
		/// </summary>
		/// <param name="source"></param>
		/// <param name="rowNum"></param>
		/// <param name="instantText"></param>
		protected override void UpdateUI(CurrencyManager source,int rowNum, string instantText)
		{
			_combo.Text = GetText(GetColumnValueAtRow(source, rowNum));
			//
			if(instantText!=null)
			{
				_combo.Text = instantText;
			}
		}
											 
		//----------------------------------------------------------------------
		// Helper Methods 
		//----------------------------------------------------------------------
		private int DataGridTableGridLineWidth
		{
			get
			{
				if(this.DataGridTableStyle.GridLineStyle == DataGridLineStyle.Solid) 
				{ 
					return 1;
				}
				else
				{
					return 0;
				}
			}
		}

		/// <summary>
		/// EndEdit
		/// </summary>
		public void EndEdit()
		{
			_inEdit = false;
			Invalidate();
		}

		private string GetText(object val)
		{
			if(val==System.DBNull.Value)
			{
				return NullText;
			}
			if(val!=null)
			{
				return val.ToString();
			}
			else
			{
				return string.Empty;
			}
		}

		private void HideComboBox()
		{
			if(_combo.Focused)
			{
				this.DataGridTableStyle.DataGrid.Focus();
			}
		   _combo.Visible = false;
		}

		private void RollBack()
		{
			_combo.Text = _oldVal;
		}

		private void PaintText(Graphics g, Rectangle bounds, string text, bool alignToRight)
		{
			Brush backBrush = new SolidBrush(this.DataGridTableStyle.BackColor);
			Brush foreBrush= new SolidBrush(this.DataGridTableStyle.ForeColor);
			PaintText(g, bounds, text, backBrush, foreBrush, alignToRight);
		}

		private void PaintText(Graphics g, Rectangle textBounds, string text, Brush backBrush, Brush foreBrush, bool alignToRight)
		{	
			Rectangle rect = textBounds;
			RectangleF rectF  = rect; 
			StringFormat format = new StringFormat();
			if(alignToRight)
			{
				format.FormatFlags = StringFormatFlags.DirectionRightToLeft;
			}
			switch(this.Alignment)
			{
				case HorizontalAlignment.Left:
					format.Alignment = StringAlignment.Near;
					break;
				case HorizontalAlignment.Right:
					format.Alignment = StringAlignment.Far;
					break;
				case HorizontalAlignment.Center:
					format.Alignment = StringAlignment.Center;
					break;
			}
			format.FormatFlags = format.FormatFlags;
			format.FormatFlags = StringFormatFlags.NoWrap;
			g.FillRectangle(backBrush, rect);
			rect.Offset(0, _yMargin);
			rect.Height -= _yMargin;
			g.DrawString(text, this.DataGridTableStyle.DataGrid.Font, foreBrush, rectF, format);
			format.Dispose();
		}
	}
}